<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    public function dashboard()
    {
        $stats = [
            'total_orders' => Order::count(),
            'total_revenue' => Order::where('status', 'delivered')->sum('total_amount'),
            'total_customers' => User::where('is_admin', false)->count(),
            'total_products' => Product::count(),
            'recent_orders' => Order::with('user')->latest()->take(5)->get(),
            'low_stock_products' => Product::where('stock_quantity', '<', 10)->take(5)->get(),
            'sales_by_category' => Category::withCount(['products' => function($query) {
                $query->whereHas('orderItems');
            }])->get(),
            'monthly_sales' => Order::where('payment_status', 'paid')
                ->where('created_at', '>=', now()->subMonths(6))
                ->select(
                    DB::raw('MONTH(created_at) as month'),
                    DB::raw('YEAR(created_at) as year'),
                    DB::raw('SUM(total_amount) as total')
                )
                ->groupBy('year', 'month')
                ->get()
        ];

        return view('admin.dashboard', compact('stats'));
    }

    public function orders(Request $request)
    {
        $query = Order::with(['user', 'orderItems']);

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('payment_status') && $request->payment_status) {
            $query->where('payment_status', $request->payment_status);
        }

        $orders = $query->latest()->paginate(20);

        return view('admin.orders.index', compact('orders'));
    }

    public function orderShow(Order $order)
    {
        $order->load(['user', 'orderItems.product']);

        return view('admin.orders.show', compact('order'));
    }

    public function orderUpdateStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|in:pending,processing,shipped,delivered,cancelled',
            'payment_status' => 'required|in:pending,paid,failed,refunded'
        ]);

        $order->update($request->only(['status', 'payment_status']));

        return redirect()->back()->with('success', 'Order status updated successfully');
    }

    public function users()
    {
        $users = User::where('is_admin', false)
            ->withCount('orders')
            ->latest()
            ->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    public function categories()
    {
        $categories = Category::withCount('products')->get();
        return view('admin.categories.index', compact('categories'));
    }

    public function categoryStore(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:categories',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:255'
        ]);

        Category::create($request->all());

        return redirect()->back()->with('success', 'Category created successfully');
    }

    public function categoryUpdate(Request $request, Category $category)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:categories,slug,' . $category->id,
            'description' => 'nullable|string',
            'is_active' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:255'
        ]);

        $category->update($request->all());

        return redirect()->back()->with('success', 'Category updated successfully');
    }

    public function categoryDestroy(Category $category)
    {
        if ($category->products()->exists()) {
            return redirect()->back()->with('error', 'Cannot delete category with associated products');
        }

        $category->delete();

        return redirect()->back()->with('success', 'Category deleted successfully');
    }

    public function reports()
    {
        $salesReport = Order::where('payment_status', 'paid')
            ->where('created_at', '>=', now()->subMonths(12))
            ->select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
                DB::raw('COUNT(*) as total_orders'),
                DB::raw('SUM(total_amount) as total_revenue')
            )
            ->groupBy('month')
            ->get();

        $topProducts = Product::withCount('orderItems')
            ->orderBy('order_items_count', 'desc')
            ->take(10)
            ->get();

        $topCategories = Category::withCount(['products' => function($query) {
            $query->whereHas('orderItems');
        }])
        ->orderBy('products_count', 'desc')
        ->take(5)
        ->get();

        return view('admin.reports.index', compact('salesReport', 'topProducts', 'topCategories'));
    }
}
