<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with('category');

        if ($request->has('category') && $request->category) {
            $query->where('category_id', $request->category);
        }

        if ($request->has('status') && $request->status !== '') {
            $query->where('is_active', $request->status);
        }

        if ($request->has('search') && $request->search) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'LIKE', "%{$searchTerm}%")
                  ->orWhere('description', 'LIKE', "%{$searchTerm}%");
            });
        }

        // Sorting
        $sort = $request->get('sort', 'latest');
        switch ($sort) {
            case 'price_asc':
                $query->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'featured':
                $query->orderBy('is_featured', 'desc')->orderBy('created_at', 'desc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(20);
        $categories = Category::active()->get();

        return view('admin.products.index', compact('products', 'categories'));
    }

    public function create()
    {
        $categories = Category::active()->get();
        $vendors = \App\Models\Vendor::all();
        $tags = \App\Models\Tag::all();
        return view('admin.products.create', compact('categories', 'vendors', 'tags'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|array',
            'name.en' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'vendor_id' => 'nullable|exists:vendors,id',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
            'price' => 'required|numeric|min:0',
            'stock_quantity' => 'required|integer|min:0',
            'description' => 'nullable|array',
            'short_description' => 'nullable|array',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'discount_type' => 'nullable|in:percent,fixed',
            'discount_value' => 'nullable|numeric|min:0',
            'discount_start' => 'nullable|date',
            'discount_end' => 'nullable|date|after_or_equal:discount_start',
        ]);

        $data = $request->except('_token', 'tags', 'images', 'main_image_index');
        $data['slug'] = Str::slug($request->name['en']);
        $data['is_active'] = $request->boolean('is_active');
        $data['discount_type'] = $request->input('discount_type');
        $data['discount_value'] = $request->input('discount_value');
        $data['discount_start'] = $request->input('discount_start');
        $data['discount_end'] = $request->input('discount_end');
        $data['name'] = $request->input('name');
        $data['description'] = $request->input('description');
        $data['short_description'] = $request->input('short_description');

        // Generate a unique slug from the English name
        $baseSlug = Str::slug($request->name['en']);
        $slug = $baseSlug;
        $counter = 2;
        while (Product::where('slug', $slug)->exists()) {
            $slug = $baseSlug . '-' . $counter++;
        }
        $data['slug'] = $slug;

        $product = Product::create($data);
        if ($request->has('tags')) {
            $product->tags()->sync($request->tags);
        }

        // Handle multiple images
        if ($request->hasFile('images')) {
            $mainIndex = (int) $request->input('main_image_index', 0);
            foreach ($request->file('images') as $idx => $image) {
                $filename = Str::slug($request->name['en']) . '-' . time() . '-' . $idx . '.webp';
                $manager = new ImageManager(new Driver());
                $img = $manager->read($image);
                Storage::disk('public')->put('products/' . $filename, (string) $img->toWebp(100));
                $product->images()->create([
                    'image_path' => 'products/' . $filename,
                    'is_main' => $idx === $mainIndex,
                ]);
            }
        }

        return redirect()->route('admin.products.index')->with('success', 'Product created successfully.');
    }

    public function show(Product $product)
    {
        $product->load('category', 'orderItems.order', 'images');

        return view('admin.products.show', compact('product'));
    }

    public function edit(Product $product)
    {
        $categories = Category::active()->get();
        $vendors = \App\Models\Vendor::all();
        $tags = \App\Models\Tag::all();
        return view('admin.products.edit', compact('product', 'categories', 'vendors', 'tags'));
    }

    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required|array',
            'name.en' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'vendor_id' => 'nullable|exists:vendors,id',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
            'price' => 'required|numeric|min:0',
            'stock_quantity' => 'required|integer|min:0',
            'description' => 'nullable|array',
            'short_description' => 'nullable|array',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'discount_type' => 'nullable|in:percent,fixed',
            'discount_value' => 'nullable|numeric|min:0',
            'discount_start' => 'nullable|date',
            'discount_end' => 'nullable|date|after_or_equal:discount_start',
        ]);

        $data = $request->except('_token', '_method', 'tags', 'images', 'main_new_image_index', 'main_image_id', 'delete_images');
        $data['slug'] = Str::slug($request->name['en']);
        $data['is_active'] = $request->boolean('is_active');
        $data['discount_type'] = $request->input('discount_type');
        $data['discount_value'] = $request->input('discount_value');
        $data['discount_start'] = $request->input('discount_start');
        $data['discount_end'] = $request->input('discount_end');
        $data['name'] = $request->input('name');
        $data['description'] = $request->input('description');
        $data['short_description'] = $request->input('short_description');

        $product->update($data);
        if ($request->has('tags')) {
            $product->tags()->sync($request->tags);
        } else {
            $product->tags()->sync([]);
        }

        // Delete images marked for removal
        if ($request->has('delete_images')) {
            foreach ($request->input('delete_images', []) as $imgId) {
                $img = $product->images()->find($imgId);
                if ($img) {
                    Storage::disk('public')->delete($img->image_path);
                    $img->delete();
                }
            }
        }

        // Handle new image uploads
        $newImages = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $idx => $image) {
                $filename = Str::slug($request->name['en']) . '-' . time() . '-' . $idx . '.webp';
                $manager = new ImageManager(new Driver());
                $img = $manager->read($image);
                Storage::disk('public')->put('products/' . $filename, (string) $img->toWebp(100));
                $newImage = $product->images()->create([
                    'image_path' => 'products/' . $filename,
                    'is_main' => false, // set below
                ]);
                $newImages[] = $newImage;
            }
        }

        // Set main image
        $mainImageId = $request->input('main_image_id');
        $mainNewImageIndex = $request->input('main_new_image_index');
        // Reset all to not main
        $product->images()->update(['is_main' => false]);
        if ($mainImageId) {
            $main = $product->images()->find($mainImageId);
            if ($main) {
                $main->is_main = true;
                $main->save();
            }
        } elseif ($mainNewImageIndex !== null && $mainNewImageIndex !== '') {
            $mainNewImageIndex = (int) $mainNewImageIndex;
            if (isset($newImages[$mainNewImageIndex])) {
                $newImages[$mainNewImageIndex]->is_main = true;
                $newImages[$mainNewImageIndex]->save();
            }
        } else {
            // If no main selected, set first image as main if exists
            $first = $product->images()->first();
            if ($first) {
                $first->is_main = true;
                $first->save();
            }
        }

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully');
    }

    public function destroy(Product $product)
    {
        // Check if product has orders
        if ($product->orderItems()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'This product cannot be deleted because it is part of one or more existing orders.',
            ], 422);
        }

        // Delete images
        if ($product->image) {
            Storage::disk('public')->delete($product->image);
        }

        if ($product->gallery) {
            foreach ($product->gallery as $image) {
                Storage::disk('public')->delete($image);
            }
        }

        $product->delete();

        return response()->json(['success' => true]);
    }

    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'products' => 'required|array',
            'products.*' => 'exists:products,id',
        ]);

        $products = Product::whereIn('id', $request->products);

        switch ($request->action) {
            case 'activate':
                $products->update(['is_active' => true]);
                break;
            case 'deactivate':
                $products->update(['is_active' => false]);
                break;
            case 'delete':
                foreach ($products->get() as $product) {
                    if ($product->orderItems()->count() == 0) {
                        if ($product->image) {
                            Storage::disk('public')->delete($product->image);
                        }
                        if ($product->gallery) {
                            foreach ($product->gallery as $image) {
                                Storage::disk('public')->delete($image);
                            }
                        }
                        $product->delete();
                    }
                }
                break;
        }

        return response()->json(['success' => true]);
    }
}
