<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

class VendorController extends Controller
{
    public function index()
    {
        $vendors = Vendor::latest()->paginate(20);
        return view('admin.vendors.index', compact('vendors'));
    }

    public function create()
    {
        return view('admin.vendors.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:vendors,name',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'description' => 'nullable|string',
        ]);

        $data = [
            'name' => $request->name,
            'description' => $request->description,
        ];

        if ($request->hasFile('logo')) {
            $image = $request->file('logo');
            $filename = Str::slug($request->name) . '-' . time() . '.webp';

            $manager = new ImageManager(new Driver());
            $img = $manager->read($image);
            Storage::disk('public')->put('vendors/' . $filename, (string)$img->toWebp(80));

            $data['logo'] = 'vendors/' . $filename;
        }

        Vendor::create($data);

        return redirect()->route('admin.vendors.index')->with('success', 'Vendor created successfully.');
    }

    public function edit(Vendor $vendor)
    {
        return view('admin.vendors.edit', compact('vendor'));
    }

    public function update(Request $request, Vendor $vendor)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:vendors,name,' . $vendor->id,
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'description' => 'nullable|string',
        ]);

        $data = [
            'name' => $request->name,
            'description' => $request->description,
        ];

        if ($request->hasFile('logo')) {
            // Delete old logo
            if ($vendor->logo) {
                Storage::disk('public')->delete($vendor->logo);
            }
            $image = $request->file('logo');
            $filename = Str::slug($request->name) . '-' . time() . '.webp';

            $manager = new ImageManager(new Driver());
            $img = $manager->read($image);
            Storage::disk('public')->put('vendors/' . $filename, (string)$img->toWebp(80));

            $data['logo'] = 'vendors/' . $filename;
        } elseif ($request->input('remove_logo') == '1' && $vendor->logo) {
            Storage::disk('public')->delete($vendor->logo);
            $data['logo'] = null;
        }

        $vendor->update($data);

        return redirect()->route('admin.vendors.index')->with('success', 'Vendor updated successfully.');
    }

    public function destroy(Vendor $vendor)
    {
        if ($vendor->products()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'This vendor cannot be deleted because it is still associated with products.'
            ], 422);
        }

        if ($vendor->logo) {
            Storage::disk('public')->delete($vendor->logo);
        }

        $vendor->delete();

        return response()->json(['success' => true]);
    }
} 