<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $currency = $request->get('currency', session('currency', 'USD'));
        session(['currency' => $currency]);

        $query = Product::active()->with(['category', 'vendor', 'tags']);

        // Search filter
        if ($request->filled('search')) {
            $searchTerm = $request->get('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('description', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('short_description', 'LIKE', "%{$searchTerm}%")
                    ->orWhereHas('category', function ($categoryQuery) use ($searchTerm) {
                        $categoryQuery->where('name', 'LIKE', "%{$searchTerm}%");
                    })
                    ->orWhereHas('vendor', function ($vendorQuery) use ($searchTerm) {
                        $vendorQuery->where('name', 'LIKE', "%{$searchTerm}%");
                    })
                    ->orWhereHas('tags', function ($tagQuery) use ($searchTerm) {
                        $tagQuery->where('name', 'LIKE', "%{$searchTerm}%");
                    });
            });
        }

        // Category filter (support multiple)
        if ($request->has('category') && $request->category) {
            $categories = is_array($request->category) ? $request->category : [$request->category];
            $categoryIds = \App\Models\Category::whereIn('slug', $categories)->pluck('id');
            if ($categoryIds->count()) {
                $query->whereIn('category_id', $categoryIds);
            }
        }

        // Vendor filter (support multiple)
        if ($request->has('vendor') && $request->vendor) {
            $vendors = is_array($request->vendor) ? $request->vendor : [$request->vendor];
            $query->whereIn('vendor_id', $vendors);
        }

        // Tag filter (support multiple)
        if ($request->has('tags') && $request->tags) {
            $tags = is_array($request->tags) ? $request->tags : [$request->tags];
            $query->whereHas('tags', function($q) use ($tags) {
                $q->whereIn('tags.id', $tags);
            });
        }

        // Price filter - Updated to use new discount logic
        if ($request->filled('min_price') || $request->filled('max_price')) {
            $query->where(function ($q) use ($request) {
                // Products with active discounts
                $q->where(function ($q2) use ($request) {
                    $q2->whereNotNull('discount_type')
                       ->where('discount_value', '>', 0)
                       ->where(function ($q3) {
                           $q3->whereNull('discount_start')
                               ->orWhere('discount_start', '<=', now());
                       })
                       ->where(function ($q3) {
                           $q3->whereNull('discount_end')
                               ->orWhere('discount_end', '>=', now());
                       });
                    
                    // Calculate final price for discounted products
                    if ($request->filled('min_price')) {
                        $q2->where(function ($q3) use ($request) {
                            // For percent discount
                            $q3->where(function ($q4) use ($request) {
                                $q4->where('discount_type', 'percent')
                                   ->whereRaw('ROUND(price * (1 - discount_value / 100), 2) >= ?', [$request->min_price]);
                            });
                            // For fixed discount
                            $q3->orWhere(function ($q4) use ($request) {
                                $q4->where('discount_type', 'fixed')
                                   ->whereRaw('GREATEST(0, price - discount_value) >= ?', [$request->min_price]);
                            });
                        });
                    }
                    
                    if ($request->filled('max_price')) {
                        $q2->where(function ($q3) use ($request) {
                            // For percent discount
                            $q3->where(function ($q4) use ($request) {
                                $q4->where('discount_type', 'percent')
                                   ->whereRaw('ROUND(price * (1 - discount_value / 100), 2) <= ?', [$request->max_price]);
                            });
                            // For fixed discount
                            $q3->orWhere(function ($q4) use ($request) {
                                $q4->where('discount_type', 'fixed')
                                   ->whereRaw('GREATEST(0, price - discount_value) <= ?', [$request->max_price]);
                            });
                        });
                    }
                });
                
                // Products without discounts (use regular price)
                $q->orWhere(function ($q2) use ($request) {
                    $q2->where(function ($q3) {
                        $q3->whereNull('discount_type')
                           ->orWhere('discount_value', '<=', 0)
                           ->orWhere('discount_start', '>', now())
                           ->orWhere('discount_end', '<', now());
                    });
                    
                    if ($request->filled('min_price')) {
                        $q2->where('price', '>=', $request->min_price);
                    }
                    if ($request->filled('max_price')) {
                        $q2->where('price', '<=', $request->max_price);
                    }
                });
            });
        }

        // Sort
        $sort = $request->get('sort', 'latest');
        switch ($sort) {
            case 'price_asc':
                // Sort by final price (discounted price if available, otherwise original price)
                $query->orderByRaw('
                    CASE 
                        WHEN discount_type IS NOT NULL 
                        AND discount_value > 0 
                        AND (discount_start IS NULL OR discount_start <= NOW())
                        AND (discount_end IS NULL OR discount_end >= NOW())
                        THEN 
                            CASE 
                                WHEN discount_type = "percent" THEN ROUND(price * (1 - discount_value / 100), 2)
                                WHEN discount_type = "fixed" THEN GREATEST(0, price - discount_value)
                                ELSE price
                            END
                        ELSE price
                    END ASC
                ');
                break;
            case 'price_desc':
                // Sort by final price (discounted price if available, otherwise original price)
                $query->orderByRaw('
                    CASE 
                        WHEN discount_type IS NOT NULL 
                        AND discount_value > 0 
                        AND (discount_start IS NULL OR discount_start <= NOW())
                        AND (discount_end IS NULL OR discount_end >= NOW())
                        THEN 
                            CASE 
                                WHEN discount_type = "percent" THEN ROUND(price * (1 - discount_value / 100), 2)
                                WHEN discount_type = "fixed" THEN GREATEST(0, price - discount_value)
                                ELSE price
                            END
                        ELSE price
                    END DESC
                ');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'featured':
                $query->orderBy('is_featured', 'desc')->orderBy('created_at', 'desc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12);
        $categories = Category::active()->withCount('products')->get();
        $vendors = \App\Models\Vendor::all();
        $tags = \App\Models\Tag::all();

        return view('product.index', compact('products', 'categories', 'vendors', 'tags', 'currency'));
    }

    public function show(Product $product, Request $request)
    {
        if (! $product->is_active) {
            abort(404);
        }

        $product->load(['images', 'category', 'vendor', 'tags']);

        $currency = $request->get('currency', session('currency', 'USD'));
        session(['currency' => $currency]);

        $relatedProducts = Product::active()
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->take(4)
            ->get();

        return view('product.show', compact('product', 'relatedProducts', 'currency'));
    }

    public function category(Category $category, Request $request)
    {
        $currency = $request->get('currency', session('currency', 'USD'));
        session(['currency' => $currency]);

        $query = $category->products()->active()->with(['vendor', 'tags']);

        // Apply filters similar to main product index
        if ($request->filled('search')) {
            $searchTerm = $request->get('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('description', 'LIKE', "%{$searchTerm}%");
            });
        }

        // Price filter
        if ($request->filled('min_price')) {
            $query->where('price', '>=', $request->min_price);
        }
        if ($request->filled('max_price')) {
            $query->where('price', '<=', $request->max_price);
        }

        // Sort
        $sort = $request->get('sort', 'latest');
        switch ($sort) {
            case 'price_asc':
                $query->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12);
        $categories = Category::active()->withCount('products')->get();

        return view('product.category', compact('category', 'products', 'categories', 'currency'));
    }

    public function apiIndex(Request $request)
    {
        $currency = $request->get('currency', 'USD');
        
        $products = Product::active()
            ->with(['category', 'vendor'])
            ->paginate(10);

        $products->getCollection()->transform(function ($product) use ($currency) {
            return [
                'id' => $product->id,
                'name' => $product->name,
                'price' => $product->getPriceInCurrency($currency),
                'final_price' => $product->getFinalPriceInCurrency($currency),
                'currency' => $currency,
                'currency_symbol' => $product->getCurrencySymbol($currency),
                'category' => $product->category->name ?? null,
                'image' => $product->getImageUrlAttribute(),
            ];
        });

        return response()->json($products);
    }

    public function apiShow(Product $product, Request $request)
    {
        $currency = $request->get('currency', 'USD');
        
        return response()->json([
            'id' => $product->id,
            'name' => $product->name,
            'description' => $product->description,
            'price' => $product->getPriceInCurrency($currency),
            'final_price' => $product->getFinalPriceInCurrency($currency),
            'currency' => $currency,
            'currency_symbol' => $product->getCurrencySymbol($currency),
            'category' => $product->category->name ?? null,
            'image' => $product->getImageUrlAttribute(),
            'in_stock' => $product->inStock(),
            'is_discounted' => $product->isDiscounted(),
        ]);
    }
}
