<?php

namespace App\Services;

class CurrencyService
{
    private $exchangeRates = [
        'USD' => [
            'AZN' => 1.70, // 1 USD = 1.70 AZN (approximate rate)
        ],
        'AZN' => [
            'USD' => 0.59, // 1 AZN = 0.59 USD (approximate rate)
        ]
    ];

    private $symbols = [
        'USD' => '$',
        'AZN' => '₼',
    ];

    /**
     * Convert amount from one currency to another
     */
    public function convert($amount, $fromCurrency, $toCurrency): float
    {
        if ($fromCurrency === $toCurrency) {
            return $amount;
        }

        if (isset($this->exchangeRates[$fromCurrency][$toCurrency])) {
            return round($amount * $this->exchangeRates[$fromCurrency][$toCurrency], 2);
        }

        return $amount; // Return original amount if conversion not available
    }

    /**
     * Get currency symbol
     */
    public function getSymbol($currency): string
    {
        return $this->symbols[$currency] ?? '$';
    }

    /**
     * Get available currencies
     */
    public function getAvailableCurrencies(): array
    {
        return array_keys($this->symbols);
    }

    /**
     * Get exchange rates
     */
    public function getExchangeRates(): array
    {
        return $this->exchangeRates;
    }

    /**
     * Update exchange rate (for admin use)
     */
    public function updateExchangeRate($fromCurrency, $toCurrency, $rate): void
    {
        $this->exchangeRates[$fromCurrency][$toCurrency] = $rate;
    }

    /**
     * Format price with currency symbol
     */
    public function formatPrice($amount, $currency): string
    {
        $symbol = $this->getSymbol($currency);
        return $symbol . number_format($amount, 2);
    }
} 