@extends('layouts.app')
@section('content')
<div class="bg-white dark:bg-gray-900">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-x-12">
            <!-- Product Image Gallery -->
            <div>
                <div class="relative w-full overflow-hidden rounded-lg bg-gray-200" style="max-width: 500px; margin: 0 auto;">
                    <img id="main-product-image"
                         src="{{ $product->mainImage ? asset('storage/' . $product->mainImage->image_path) : ($product->images->first() ? asset('storage/' . $product->images->first()->image_path) : 'https://via.placeholder.com/600') }}"
                         alt="{{ $product->name }}"
                         class="w-full h-auto object-contain object-center transition-all duration-200" />
                    @if($product->images->count() > 1)
                        <button id="arrow-left" type="button"
                            class="absolute left-2 top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white text-gray-700 rounded-full p-2 shadow focus:outline-none z-20">
                            <svg class="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
                            </svg>
                        </button>
                        <button id="arrow-right" type="button"
                            class="absolute right-2 top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white text-gray-700 rounded-full p-2 shadow focus:outline-none z-20">
                            <svg class="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                            </svg>
                        </button>
                    @endif
                </div>
                @if($product->images->count() > 1)
                <div class="flex gap-2 mt-4">
                    @foreach($product->images as $img)
                        <img src="{{ asset('storage/' . $img->image_path) }}" 
                             alt="{{ $product->name }} thumbnail" 
                             class="w-16 h-16 object-cover rounded border cursor-pointer thumbnail-image {{ $img->is_main ? 'ring-2 ring-indigo-500' : '' }}"
                             data-full="{{ asset('storage/' . $img->image_path) }}">
                    @endforeach
                </div>
                @endif
            </div>
            <!-- Product Info -->
            <div class="mt-8 md:mt-0">
                <h1 class="text-3xl font-bold tracking-tight text-gray-900 dark:text-white">{{ $product->getTranslation('name', app()->getLocale()) }}</h1>
                <div class="mt-4">
                    @php
                        $currency = session('currency', 'USD');
                        $price = $product->getPriceInCurrency($currency);
                        $finalPrice = $product->getFinalPriceInCurrency($currency);
                        $symbol = $product->getCurrencySymbol($currency);
                    @endphp
                    <p class="text-3xl tracking-tight text-gray-900 dark:text-white">
                        @if($product->isDiscounted())
                            <span class="line-through text-gray-500">{{ $symbol }}{{ number_format($price, 2) }}</span>
                            <span class="text-red-600 font-bold">{{ $symbol }}{{ number_format($finalPrice, 2) }}</span>
                        @else
                            <span>{{ $symbol }}{{ number_format($price, 2) }}</span>
                        @endif
                    </p>
                </div>
                @if($product->category)
                <div class="mt-4">
                    <span class="text-sm text-gray-500 dark:text-gray-400">
                        {{ __('category') }}: <a href="{{ route('products.category', $product->category) }}" class="font-medium text-indigo-600 hover:text-indigo-500">{{ $product->category ? $product->category->getTranslation('name', app()->getLocale()) : '' }}</a>
                    </span>
                </div>
                @endif
                <div class="mt-6">
                    <h3 class="text-lg font-medium text-gray-900 dark:text-white">{{ __('description') }}</h3>
                    <div class="space-y-6 text-base text-gray-700 dark:text-gray-300 mt-4">
                        @if($product->getTranslation('description', app()->getLocale()))
                            <p>{!! nl2br(e($product->getTranslation('description', app()->getLocale()))) !!}</p>
                        @else
                            <p class="text-gray-500 italic">{{ __('no_description') }}</p>
                        @endif
                    </div>
                </div>
                <div class="mt-6">
                    @if($product->inStock())
                        <p class="text-sm text-green-600 dark:text-green-400">
                            <svg class="inline-block h-5 w-5 mr-1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                <path fill-rule="evenodd" d="M16.704 4.153a.75.75 0 01.143 1.052l-8 10.5a.75.75 0 01-1.127.075l-4.5-4.5a.75.75 0 011.06-1.06l3.894 3.893 7.48-9.817a.75.75 0 011.052-.143z" clip-rule="evenodd" />
                            </svg>
                            {{ __('in_stock') }} ({{ $product->stock_quantity }} {{ __('available') }})
                        </p>
                    @else
                        <p class="text-sm text-red-600 dark:text-red-400">
                            <svg class="inline-block h-5 w-5 mr-1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                <path d="M6.28 5.22a.75.75 0 00-1.06 1.06L8.94 10l-3.72 3.72a.75.75 0 101.06 1.06L10 11.06l3.72 3.72a.75.75 0 101.06-1.06L11.06 10l3.72-3.72a.75.75 0 00-1.06-1.06L10 8.94 6.28 5.22z" />
                            </svg>
                            {{ __('out_of_stock') }}
                        </p>
                    @endif
                </div>
                @if($product->inStock())
                    <form id="add-to-cart-form" action="{{ route('cart.add') }}" method="POST" class="mt-10">
                        @csrf
                        <input type="hidden" name="product_id" value="{{ $product->id }}">
                        <div class="flex items-center space-x-4">
                            <div class="flex items-center">
                                <label for="quantity" class="sr-only">{{ __('quantity') }}</label>
                                <button type="button" class="decrement-qty px-2 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 hover:bg-gray-300 dark:hover:bg-gray-600" aria-label="Decrease quantity">-</button>
                                <input type="number" name="quantity" id="quantity" value="1" min="1" max="{{ $product->stock_quantity }}" class="w-20 rounded-md border border-gray-300 py-1.5 px-2 text-base leading-5 focus:border-indigo-500 focus:outline-none focus:ring-1 focus:ring-indigo-500 sm:text-sm dark:bg-gray-700 dark:border-gray-600 dark:text-white text-center cart-qty-input" autocomplete="off">
                                <button type="button" class="increment-qty px-2 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 hover:bg-gray-300 dark:hover:bg-gray-600" aria-label="Increase quantity">+</button>
                            </div>
                            <button type="submit" class="flex max-w-xs flex-1 items-center justify-center rounded-md border border-transparent bg-indigo-600 px-8 py-3 text-base font-medium text-white hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 focus:ring-offset-gray-50 sm:w-full">{{ __('add_to_cart') }}</button>
                        </div>
                    </form>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Related Products -->
@if($relatedProducts->count() > 0)
<div class="bg-gray-50 dark:bg-gray-800">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <h2 class="text-2xl font-bold tracking-tight text-gray-900 dark:text-white mb-8">{{ __('related_products') }}</h2>
        <div class="grid grid-cols-1 gap-y-10 gap-x-6 sm:grid-cols-2 lg:grid-cols-4 xl:gap-x-8">
            @foreach($relatedProducts as $relatedProduct)
                <x-product-card :product="$relatedProduct" />
            @endforeach
        </div>
    </div>
</div>
@endif

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    var form = document.getElementById('add-to-cart-form');
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            const button = form.querySelector('button[type="submit"]');
            const originalButtonText = button.innerHTML;
            button.disabled = true;
            button.innerHTML = 'Adding...';
            const formData = new FormData(form);
            fetch(form.action, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Accept': 'application/json',
                }
            })
            .then(async response => {
                let data;
                try {
                    data = await response.json();
                } catch (err) {
                    throw { message: 'Unexpected server response.' };
                }
                if (!response.ok) {
                    throw data;
                }
                return data;
            })
            .then(data => {
                if (data.success) {
                    updateCartCount();
                    button.innerHTML = 'Added!';
                    setTimeout(() => {
                        button.disabled = false;
                        button.innerHTML = originalButtonText;
                    }, 2000);
                } else {
                    button.disabled = false;
                    button.innerHTML = originalButtonText;
                    alert(data.message || 'An error occurred.');
                }
            })
            .catch(error => {
                button.disabled = false;
                button.innerHTML = originalButtonText;
                alert((error && error.message) || 'An error occurred while adding the item to the cart.');
            });
        });
    }
    // Product image gallery switching and arrow navigation
    const mainImg = document.getElementById('main-product-image');
    const thumbnails = Array.from(document.querySelectorAll('.thumbnail-image'));
    let currentIdx = 0;
    if (thumbnails.length > 0) {
        // Set initial index based on main image
        const mainSrc = mainImg.src;
        currentIdx = thumbnails.findIndex(t => t.getAttribute('data-full') === mainSrc);
        if (currentIdx === -1) currentIdx = 0;
    }
    function showImage(idx) {
        if (thumbnails.length === 0) return;
        currentIdx = (idx + thumbnails.length) % thumbnails.length;
        mainImg.src = thumbnails[currentIdx].getAttribute('data-full');
        thumbnails.forEach(t => t.classList.remove('ring-2', 'ring-indigo-500'));
        thumbnails[currentIdx].classList.add('ring-2', 'ring-indigo-500');
    }
    thumbnails.forEach(function(thumb, idx) {
        thumb.addEventListener('click', function() {
            showImage(idx);
        });
    });
    const leftArrow = document.getElementById('arrow-left');
    const rightArrow = document.getElementById('arrow-right');
    if (leftArrow && rightArrow) {
        leftArrow.addEventListener('click', function() {
            showImage(currentIdx - 1);
        });
        rightArrow.addEventListener('click', function() {
            showImage(currentIdx + 1);
        });
    }
    // Quantity increment/decrement logic
    const qtyInput = document.getElementById('quantity');
    const incBtn = document.querySelector('.increment-qty');
    const decBtn = document.querySelector('.decrement-qty');
    if (qtyInput && incBtn && decBtn) {
        incBtn.addEventListener('click', function() {
            let val = parseInt(qtyInput.value) || 1;
            const max = parseInt(qtyInput.getAttribute('max')) || 9999;
            if (val < max) {
                qtyInput.value = val + 1;
            }
        });
        decBtn.addEventListener('click', function() {
            let val = parseInt(qtyInput.value) || 1;
            if (val > 1) {
                qtyInput.value = val - 1;
            }
        });
        qtyInput.addEventListener('change', function() {
            let val = parseInt(qtyInput.value) || 1;
            const max = parseInt(qtyInput.getAttribute('max')) || 9999;
            if (val < 1) val = 1;
            if (val > max) val = max;
            qtyInput.value = val;
        });
    }
});
</script>
@endpush
@endsection
